/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   copyright            : (C) 2003 by Zhang Yong                         *
 *   email                : z-yong163@163.com                              *
 ***************************************************************************/

#include "tcpsession.h"
#include "icqmain.h"
#include "icqlog.h"

#pragma pack(1)
struct TCP_PACKET_HDR {
	uint16 ver;
	uint32 reserved;
	uint16 cmd;
};
#pragma pack()

enum {
	TCP_STATUS_NOT_CONN,
	TCP_STATUS_HELLO_WAIT,
	TCP_STATUS_ESTABLISHED,
	TCP_STATUS_LISTEN,
};

#define TCP_CMD_HELLO	1

const uint16 TCP_VER = 1;


TCPSession::TCPSession(ICQMain *main, Socket *sock, const char *type, const char *name, bool isSend)
{
	icqMain = main;
	tcpSocket = sock;
	sessionType = type;
	contactName = name;
	isSender = isSend;

	tcpSocket->create(SOCK_STREAM, this);
	tcpSocket->selectEvent(Socket::READ);

	listener = NULL;
	writeEnabled = false;
	isDisabled = false;
	status = TCP_STATUS_NOT_CONN;
	recvBufSize = sendBufSize = 0;
}

TCPSession::~TCPSession()
{
	if (listener)
		listener->destroy();

	delete tcpSocket;
}

void TCPSession::clearSendQueue()
{
	list<ICQOutPacket *>::iterator it;
	for (it = sendQueue.begin(); it != sendQueue.end(); ++it)
		delete *it;
	sendQueue.clear();
}

bool TCPSession::isListen()
{
	return (status == TCP_STATUS_LISTEN);
}

void TCPSession::enableWrite(bool enable)
{
	if (status != TCP_STATUS_ESTABLISHED)
		return;

	writeEnabled = enable;

	if (enable)
		tcpSocket->selectEvent(Socket::READ | Socket::WRITE);
	else if (!sendBufSize)
		tcpSocket->selectEvent(Socket::READ);
}

void TCPSession::destroy()
{
	icqMain->removeSession(this);
	delete this;
}

OutPacket *TCPSession::createPacket(uint16 cmd)
{
	static ICQOutPacket out;

	ICQOutPacket *p;

	if (status == TCP_STATUS_ESTABLISHED) {
		out.reset();
		p = &out;
	} else {
		p = new ICQOutPacket;
		sendQueue.push_back(p);
	}

	*p << (uint16) 0;
	*p << TCP_VER << (uint32) 0 << cmd;
	return p;
}

bool TCPSession::sendPacket(OutPacket *out)
{
	if (status != TCP_STATUS_ESTABLISHED)
		return false;

	ICQOutPacket *p = (ICQOutPacket *) out;

	char *data = p->data;
	int total = p->getSize();
	uint16 n = total - sizeof(n);
	*(uint16 *) data = htons(n);

	int sent = 0;

	if (!sendBufSize) {
		sent = tcpSocket->send(data, total);
		if (sent < 0) {
#ifdef _WIN32
			if (WSAGetLastError() != WSAEWOULDBLOCK)
#else
			if (errno != EWOULDBLOCK)
#endif
				return false;

			sent = 0;
		}
	}

	int left = total - sent;
	if (left > 0) {
		if (sendBufSize + left > sizeof(sendBuf))
			return false;

		memcpy(sendBuf + sendBufSize, data + sent, left);
		sendBufSize += left;

		tcpSocket->selectEvent(Socket::READ | Socket::WRITE);
	}

	return true;
}

void TCPSession::sendHello()
{
	OutPacket *out = createPacket(TCP_CMD_HELLO);
	*out << icqMain->getUserName() << isSender;
	sendPacket(out);
}

void TCPSession::connect(uint32 ip, uint16 port)
{
	status = TCP_STATUS_NOT_CONN;

	tcpSocket->selectEvent(Socket::READ | Socket::WRITE | Socket::EXCEPTION);

	in_addr addr;
	addr.s_addr = htonl(ip);
	const char *host = inet_ntoa(addr);
	tcpSocket->connect(host, port);
}

void TCPSession::listen()
{
	::listen(tcpSocket->getFd(), 5);

	status = TCP_STATUS_LISTEN;
}

void TCPSession::recvPacket()
{
	int n;

	while (true) {
		n = tcpSocket->receive(recvBuf + recvBufSize, sizeof(recvBuf) - recvBufSize);
		if (n <= 0)
			break;

		recvBufSize += n;
		char *start = recvBuf;
		char *end = start + recvBufSize;
		uint16 len;

		while (start + sizeof(len) < end) {
			len = ntohs(*(uint16 *) start);
			if (end - start - sizeof(len) < len)
				break;

			start += sizeof(len);
			onPacketReceived(start, len);
			start += len;
		}

		recvBufSize = end - start;
		if (recvBufSize > 0)
			memmove(recvBuf, start, recvBufSize);
	}

	if (n <= 0)
		onClose();
}

void TCPSession::onConnect()
{
	tcpSocket->selectEvent(Socket::READ);

	status = TCP_STATUS_ESTABLISHED;

	sendHello();

	list<ICQOutPacket *>::iterator it;
	for (it = sendQueue.begin(); it != sendQueue.end(); ++it)
		sendPacket(*it);

	clearSendQueue();

	onTCPEstablished();
}

void TCPSession::onAccept()
{
	TCPSession *s = icqMain->acceptSession(sessionType.c_str(), tcpSocket);
	if (s)
		s->status = TCP_STATUS_HELLO_WAIT;
}

void TCPSession::onClose()
{
	tcpSocket->close();

	if (listener)
		listener->onClose();
}

void TCPSession::onTCPEstablished()
{
	if (!listener) {
		listener = icqMain->getSessionListener(sessionType.c_str(), this);
		if (!listener)
			onClose();
	}
}

void TCPSession::onPacketReceived(const char *data, int n)
{
	if (n < sizeof(TCP_PACKET_HDR))
		return;
	
	TCP_PACKET_HDR header;

	ICQInPacket in(data, n);
	in >> header.ver >> header.reserved >> header.cmd;

	onPacketReceived(in, header);
}

void TCPSession::onPacketReceived(ICQInPacket &in, TCP_PACKET_HDR &header)
{
	uint16 cmd = header.cmd;

	if (cmd == TCP_CMD_HELLO)
		onHello(in);
	else if (status == TCP_STATUS_ESTABLISHED) {
		if (listener)
			listener->onReceive(in, cmd);
	}
}

void TCPSession::onHello(InPacket &in)
{
	if (status != TCP_STATUS_HELLO_WAIT)
		return;

	status = TCP_STATUS_ESTABLISHED;

	const char *name;
	in >> name >> isSender;
	isSender = !isSender;

	contactName = name;

	onTCPEstablished();
}

void TCPSession::onSocketRead()
{
	if (status == TCP_STATUS_LISTEN)
		onAccept();
	else
		recvPacket();
}

void TCPSession::onSocketWrite()
{
	if (status == TCP_STATUS_NOT_CONN) {
		onConnect();
		return;
	}

	if (sendBufSize > 0) {
		int n = tcpSocket->send(sendBuf, sendBufSize);
		if (n > 0) {
			sendBufSize -= n;
			if (sendBufSize > 0)
				memmove(sendBuf, sendBuf + n, sendBufSize);
		}
	}

	if (!sendBufSize && writeEnabled)
		listener->onSend();
	else
		tcpSocket->selectEvent(Socket::READ);
}

void TCPSession::onSocketException()
{
	ICQ_LOG("Can not connect to contact %s.\n", contactName.c_str());

	clearSendQueue();

	isDisabled = true;
}
