/*
 * GTK See -- an image viewer based on GTK+
 * Copyright (C) 1998 Hotaru Lee <jkhotaru@mail.sti.com.cn> <hotaru@163.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <gtk/gtkprivate.h>

#include "intl.h"
#include "common_tools.h"
#include "detect.h"
#include "dndviewport.h"
#include "gtksee.h"
#include "gtypes.h"
#include "imagelist.h"
#include "rc.h"
#include "scanline.h"
#include "savefile.h"
#include "viewer.h"
#include "viewerstatus.h"
#include "viewertoolbar.h"
#include "dialog.h"

#include "../icons/gtksee.xpm"
#include "pixmaps/blank.xpm"

#define MAX_WIDTH (gdk_screen_width()-10)
#define MAX_HEIGHT (gdk_screen_height()-80)
#define MAX_FULL_WIDTH (gdk_screen_width())
#define MAX_FULL_HEIGHT (gdk_screen_height())
#define MOVEMENT_INCREASE 10

static GtkWidget  *window, *toolbar, *statusbar, *imagelist;
static GList      *image_infos;
static ImageInfo  *current;
static GtkWidget  *frame;
static gchar      viewing_image[256];
static gboolean   viewer_busy;
static GdkCursor  *viewer_watch_cursor = NULL;
static GtkWidget  *viewport, *viewport_image;
static GtkWidget  *sshow_item;

static gboolean   slide_show = FALSE;
static guint      timeout_callback_tag = 0;
static gboolean   have_xgrab;

void view_current_image                ();
void viewer_set_busy_cursor            ();
void viewer_set_normal_cursor          ();
void viewer_key_press                  (GtkWidget *widget, GdkEvent *event,
                                          gpointer data);
void viewer_key_release                (GtkWidget *widget, GdkEvent *event,
                                          gpointer data);
void viewer_menu_popup                 (GtkWidget *widget, gpointer data);
void viewer_popup_view                 (GtkWidget *widget, gpointer data);
void viewer_popup_slideshow            (GtkWidget *widget, gpointer data);
void viewer_popup_fitscreen            (GtkWidget *widget, gpointer data);
void viewer_slideshow_next             (gpointer data);
void viewer_slideshow_toggled_internal (gboolean active);

GtkWidget*  get_viewer_window_internal ();
GtkWidget*  get_full_view_window       ();

/* Define a global var */
ImageCache *cache;


gboolean move_pointer         = FALSE;
gint     x_pos_button         = 0;
gint     y_pos_button         = 0;
gboolean slide_button_pressed = FALSE;
gint     min_image_list       = 0;
gint     max_image_list       = 0;
GtkStyle *old_style;


#define  _NET_WM_STATE_REMOVE 0
#define  _NET_WM_STATE_ADD    1
#define  _NET_WM_STATE_TOGGLE 2


static void
net_wm_state(gchar *hint, gboolean state)
{
   XEvent   xev;

   xev.type                = ClientMessage;
   xev.xclient.type        = ClientMessage;
   xev.xclient.window      = GDK_WINDOW_XWINDOW(GTK_WIDGET(window)->window);
   xev.xclient.message_type= gdk_atom_intern("_NET_WM_STATE", FALSE);
   xev.xclient.format      = 32;
   xev.xclient.data.l[0]   = state ? _NET_WM_STATE_ADD : _NET_WM_STATE_REMOVE;
   xev.xclient.data.l[1]   = gdk_atom_intern(hint, FALSE);
   xev.xclient.data.l[2]   = 0;

   XSendEvent(GDK_DISPLAY(),
               GDK_ROOT_WINDOW(),
               False,
               SubstructureNotifyMask,
               &xev);

   return;
}

void always_on_top(gboolean state)
{
   net_wm_state("_NET_WM_STATE_ABOVE", state);
   /* Apparently KDE 3.1.0 and possibly below does not implement
    * _NET_WM_STATE_ABOVE but _NET_WM_STATE_STAYS_ON_TOP that implies
    * approximately the same thing
    */
   net_wm_state("_NET_WM_STATE_STAYS_ON_TOP", state);
}

void
repaint_viewport_image()
{
   ImageCache  *loaded_cache;

   /* Get the image to show */
   loaded_cache = scanline_get_cache();

   gdk_draw_rgb_image(viewport_image->window,
                        viewport_image->style->white_gc,
                        0, 0,
                        loaded_cache->buffer_width,
                        loaded_cache->buffer_height,
                        GDK_RGB_DITHER_MAX,
                        loaded_cache->buffer,
                        loaded_cache->buffer_width*3);

   return;
}

void
viewer_move_pointer(GtkWidget *widget, GtkAllocation *allocate, gpointer data)
{

   if(!GTK_IS_WIDGET(widget))
   {
      return;
   }

   if (move_pointer)
   {
      GdkWindow   *win;
      gint        x, y;
      GdkWindowPrivate *winprivate;
      XWindowAttributes win_attributes;
      Window child = None;
      GtkRequisition r;

gint width,height,depth;

      win = window->window;
      winprivate=(GdkWindowPrivate *)window->window;

      gdk_window_get_geometry(win, &x, &y,&width,&height,&depth);
/*
printf("Datos de allocate solicit : => x:%i\ty:%i\n", allocate->x, allocate->y);

      printf("gdk_window_get_geometry   : => x:%i\ty:%i\twidth: %i\theight: %i\n", x, y,width, height);
      gdk_window_get_root_origin(win, &x, &y);
      printf("gdk_window_get_root_origin: => x:%i\ty:%i\n", x, y);
      gdk_window_get_origin(win, &x, &y);
      printf("gdk_window_get_origin     : => x:%i\ty:%i\n", x, y);

      gdk_window_get_origin(win, &x, &y);

      x += x_pos_button;
      y += y_pos_button;

      while (gtk_events_pending()) gtk_main_iteration();

      XWarpPointer(GDK_DISPLAY(),
                     None,
                     GDK_ROOT_WINDOW(),
                     0, 0, 0, 0,
                     (int)x, (int)y);
*/

      XGetWindowAttributes(winprivate->xdisplay, winprivate->xwindow,
                              &win_attributes);

      XTranslateCoordinates(winprivate->xdisplay, winprivate->xwindow,
                              win_attributes.root,
                              -win_attributes.border_width,
                              -win_attributes.border_width,
                              &x, &y, &child);
      printf("XTranslateCoordinates     : => x:%i\ty:%i\n", x, y);
      gtk_widget_size_request(window, &r);
      printf("Window Request            : => x:%i\ty:%i\n", r.width, r.height);
      XWarpPointer(winprivate->xdisplay,
                     None,
                     winprivate->xwindow,
                     0, 0, 0, 0,
                     (int)x_pos_button, (int)y_pos_button);

      //printf("Despues de Viewer_Move_Pointer==> x:%i\ty:%i\n", x, y);
      move_pointer = FALSE;
   }
}

void
viewer_set_busy_cursor()
{
   if (!viewer_busy)
   {
      viewer_busy = TRUE;
      gdk_window_set_cursor(window->window, viewer_watch_cursor);
      while (gtk_events_pending()) gtk_main_iteration();
   }
}

void
viewer_set_normal_cursor()
{
   if (viewer_busy)
   {
      viewer_busy = FALSE;
      gdk_window_set_cursor(window->window, NULL);
   }
}

void
viewer_set_slide_show(gboolean s)
{
   slide_show = s;
}

void
view_current_image()
{
   gboolean    too_big, need_scale;
   gchar       buffer[512], *ptr;
   gint        width, height, max_width, max_height, tmp;
   gint        value;
   gboolean    full_screen, fit_screen;
   ImageCache  *loaded_cache;

   if (viewer_busy) return;

   /* setting window title */
   if (imagelist == NULL)
   {
      ptr = strrchr(current->name, '/');
      if (ptr == NULL)
         ptr = current->name;
      else
         ptr++;
   } else
      ptr = current->name;

   if (rc_get_boolean("show_statusbar") == FALSE)
   {
      gint actual, total;

      actual= (min_image_list) ? current->serial : current->serial + 1;
      total = (min_image_list) ? max_image_list  : max_image_list  + 1;
      sprintf(buffer, "%s [%i/%i]- GTK See", ptr, actual, total);
   } else
   {
      sprintf(buffer, "%s - GTK See", ptr);
   }

   gtk_window_set_title(GTK_WINDOW(window), buffer);

   full_screen = rc_get_boolean("full_screen");
   fit_screen  = rc_get_boolean("fit_screen");
/*
   if (!current -> valid)
   {
      if (!full_screen)
      {
         if (imagelist == NULL)
         {
            set_viewer_status_type(image_type_get_pixmap(current->type), image_type_get_mask(current->type));
         } else
         {
            set_viewer_status_type(current->type_pixmap, current->type_mask);
         }
         set_viewer_status_name  (ptr);
         set_viewer_status_size  (current->size);
         set_viewer_status_prop  (_("Invalid"));
         set_viewer_status_zoom  (100);
         set_viewer_status_number(((min_image_list) ? current->serial : current->serial + 1),
                                  ((min_image_list) ? max_image_list  : max_image_list  + 1)
                                 );
      }
      viewer_set_normal_cursor();
      return;
   }
*/
   /* setting status bar */
   if (!full_screen)
   {
      if (imagelist == NULL)
      {
         set_viewer_status_type(image_type_get_pixmap(current->type), image_type_get_mask(current->type));
      } else
      {
         set_viewer_status_type(current->type_pixmap, current->type_mask);
      }
      set_viewer_status_name  (ptr);
      set_viewer_status_size  (current->size);
      set_viewer_status_number(((min_image_list) ? current->serial : current->serial + 1),
                               ((min_image_list) ? max_image_list  : max_image_list  + 1)
                              );

      if (!current -> valid)
      {
         set_viewer_status_prop  (_("Invalid"));
         set_viewer_status_zoom  (100);
         return ;
      }

      image_convert_info      (current, buffer);
      set_viewer_status_prop  (buffer);
   }

   viewer_set_busy_cursor();

   if (imagelist != NULL)
   {
      strcpy(viewing_image, image_list_get_dir(IMAGE_LIST(imagelist)));
      if (viewing_image[strlen(viewing_image) - 1] != '/')
      {
         strcat(viewing_image,"/");
      }
      strcat(viewing_image, current->name);
   } else
   {
      strcpy(viewing_image, current->name);
   }

   if (current->width < 0)
   {
      if (detect_image_type(viewing_image, current))
      {
         /* we should refresh the image info */
         if (!full_screen)
         {
            image_convert_info(current, buffer);
            set_viewer_status_prop(buffer);
         }
         if (imagelist != NULL)
         {
            image_list_update_info(IMAGE_LIST(imagelist), current);
         }
      }
   }

   max_width   = (full_screen ? MAX_FULL_WIDTH : MAX_WIDTH );
   max_height  = (full_screen ? MAX_FULL_HEIGHT: MAX_HEIGHT);

   width    = current->width;
   height   = current->height;
   too_big  = (width > max_width || height > max_height);

   need_scale = (fit_screen && too_big);

   if (need_scale)
   {
      if (max_width * height < max_height * width)
      {
         if (!full_screen)
         {
            set_viewer_status_zoom(max_width * 100 / width);
         }
         tmp   = width;
         width = max_width;
         height= width * height / tmp;
      } else
      {
         if (!full_screen)
         {
            set_viewer_status_zoom(max_height * 100 / height);
         }
         tmp   = height;
         height= max_height;
         width = height * width / tmp;
      }
   } else
   {
      if (!full_screen)
      {
         set_viewer_status_zoom(100);
      }
      width = min(width , max_width);
      height= min(height, max_height);
   }

   if (!need_scale)
   {
      max_width = -1;
      max_height= -1;
   }

   /* for slideshow: disable timeout temporarily before loading */
   if (slide_show && timeout_callback_tag != 0)
   {
      gtk_timeout_remove(timeout_callback_tag);
   }

   load_scaled_image(viewing_image, current,
                     max_width, max_height,
                     window,
                     viewport_image, 0);

   /* Get the image to show */
   loaded_cache = scanline_get_cache();

   /* Resize the drawing area */
   gtk_preview_size(GTK_PREVIEW(viewport_image),
                     loaded_cache->buffer_width,
                     loaded_cache->buffer_height);

   /* Resize the container window */
   gtk_widget_set_usize(viewport, max_width, max_height);

   /* The expose_event will draw the image ... */

   /* allow the user to drag the viewport while displaying */
   gtk_grab_add(viewport);

   /* reset the viewport position to (0,0) */
   dnd_viewport_reset(DND_VIEWPORT(viewport));

   while (gtk_events_pending()) gtk_main_iteration();

   gtk_grab_remove(viewport);

   if (slide_show)
   {
      value = rc_get_int("slideshow_delay");
      timeout_callback_tag = gtk_timeout_add(
         ((value == RC_NOT_EXISTS) ? SLIDESHOW_DELAY : value),
         (GtkFunction)viewer_slideshow_next, NULL);
   }

   viewer_set_normal_cursor();

   return;
}

void
viewer_toolbar_browse(GtkWidget *widget, gpointer data)
{
   if (viewer_busy) return;

   /* Put in order... */

   move_pointer         = FALSE;
   x_pos_button         = 0;
   y_pos_button         = 0;
   slide_button_pressed = FALSE;

   if (slide_show) gtk_timeout_remove(timeout_callback_tag);
   viewer_next_enable(TRUE);
   viewer_prev_enable(TRUE);

   if (have_xgrab)
   {
      gdk_pointer_ungrab(0);
      gdk_keyboard_ungrab(0);
   }

   if (rc_get_int("background_color") != -1)
   {
      scanline_init(gdk_color_context_new(
                        gdk_window_get_visual(window->window),
                        gdk_window_get_colormap(window->window)),
                        old_style);
   }

   gtk_widget_destroy(window);

   if (imagelist == NULL)
   {
      gchar *root;
      gint  i, len;

      root  = gtksee_malloc(BUF_STD);
      len   = strlen(current->name);

      strncpy(root, current->name, BUF_STD);
      for (i=len-1; root[i] != '/'; i--)
         ;

      if (i == 0) i = 1;

      root[i] = '\0';

      g_list_free(image_infos);
      gtksee_main(root);

   } else
   {
      show_browser();
   }
}

void
viewer_toolbar_full_screen(GtkWidget *widget, gpointer data)
{
   if (viewer_busy) return;

   rc_set_boolean("full_screen", TRUE);

   gtk_widget_destroy(window);
   get_full_view_window();
}

void
viewer_toolbar_save_image(GtkWidget *widget, gpointer data)
{
   if (viewer_busy) return;

   savefile (cache);
   viewer_save_enable(FALSE);
}

void
viewer_toolbar_next_image(GtkWidget *widget, gpointer data)
{
   GList       *tmplist;
   ImageInfo   *tmp;

   if (viewer_busy) return;

   if (imagelist == NULL)
   {
      tmplist = g_list_next(image_infos);
      if (tmplist == NULL) return;
      image_infos = tmplist;
      tmp = (ImageInfo*) image_infos->data;
   } else
   {
      tmp = image_list_get_next(IMAGE_LIST(imagelist), current);
   }

   if (tmp == NULL) return;

   current = tmp;

   if (!rc_get_boolean("full_screen"))
   {
      viewer_slideshow_stop_by_next_prev();

      if (imagelist == NULL)
      {
         viewer_next_enable(g_list_next(image_infos) != NULL);
         viewer_prev_enable(g_list_previous(image_infos) != NULL);
      } else
      {
         if (min_image_list == current->serial-1) viewer_prev_enable(TRUE);
         if (max_image_list == current->serial  ) viewer_next_enable(FALSE);

         viewer_save_enable(FALSE);
      }

      if (GTK_IS_BUTTON(widget))
      {
         x_pos_button = widget->allocation.x + (widget->allocation.width >> 1);
         y_pos_button = widget->allocation.y + (widget->allocation.height>> 1);
         move_pointer = TRUE;
      }
   }
   view_current_image();

}

void
viewer_toolbar_prev_image(GtkWidget *widget, gpointer data)
{
   GList *tmplist;
   ImageInfo *tmp;

   if (viewer_busy) return;

   if (imagelist == NULL)
   {
      tmplist = g_list_previous(image_infos);
      if (tmplist == NULL) return;
      image_infos = tmplist;
      tmp = (ImageInfo*) image_infos->data;
   } else
   {
      tmp = image_list_get_previous(IMAGE_LIST(imagelist), current);
   }

   if (tmp == NULL) return;

   current = tmp;

   if (! rc_get_boolean("full_screen"))
   {
      viewer_slideshow_stop_by_next_prev();

      if (imagelist == NULL)
      {
         viewer_next_enable(g_list_next(image_infos) != NULL);
         viewer_prev_enable(g_list_previous(image_infos) != NULL);
      } else
      {
         if (min_image_list == current->serial  ) viewer_prev_enable(FALSE);
         if (max_image_list == current->serial+1) viewer_next_enable(TRUE);
         viewer_save_enable(FALSE);
      }
      if (GTK_IS_BUTTON(widget))
      {
         x_pos_button = widget->allocation.x + (widget->allocation.width >> 1);
         y_pos_button = widget->allocation.y + (widget->allocation.height>> 1);
         move_pointer = TRUE;
      }
   }
   view_current_image();

}

void
viewer_toolbar_fitscreen_toggled(GtkWidget *widget, gpointer data)
{
   if (viewer_busy) return;

   rc_set_boolean("fit_screen", GTK_TOGGLE_BUTTON(widget)->active);

   view_current_image();
}

void
viewer_toolbar_slideshow_toggled(GtkWidget *widget, gpointer data)
{
   if (viewer_busy) return;

   viewer_slideshow_toggled_internal(GTK_TOGGLE_BUTTON(widget)->active);

}

void
viewer_slideshow_toggled_internal(gboolean active)
{
   gint value;

   slide_show = active;
   if (slide_show)
   {
      value = rc_get_int("slideshow_delay");
      timeout_callback_tag = gtk_timeout_add(
         ((value == RC_NOT_EXISTS) ? SLIDESHOW_DELAY : value),
         (GtkFunction)viewer_slideshow_next, NULL);
   } else
   {
      gtk_timeout_remove(timeout_callback_tag);
   }
}

void
viewer_slideshow_next(gpointer data)
{
   GList       *tmplist;
   ImageInfo   *tmp;
   gboolean    full_screen;

   if (viewer_busy) return;

   if (imagelist == NULL)
   {
      tmplist = g_list_next(image_infos);
      if (tmplist == NULL)
      {
         image_infos = g_list_first(image_infos);
      } else
      {
         image_infos = tmplist;
      }
      tmp = (ImageInfo*) image_infos->data;
      if (tmp == NULL)
      {
         image_infos = g_list_first(image_infos);
         current = (ImageInfo*) image_infos->data;
      } else current = tmp;
   } else
   {
      tmp = image_list_get_next(IMAGE_LIST(imagelist), current);
      if (tmp == NULL) current = image_list_get_first(IMAGE_LIST(imagelist));
      else current = tmp;
   }

   full_screen = rc_get_boolean("full_screen");
   if (!full_screen)
   {
      if (imagelist == NULL)
      {
         viewer_next_enable(g_list_next(image_infos) != NULL);
         viewer_prev_enable(g_list_previous(image_infos) != NULL);
      } else
      {
         viewer_next_enable(image_list_get_next(IMAGE_LIST(imagelist), current) != NULL);
         viewer_prev_enable(image_list_get_previous(IMAGE_LIST(imagelist), current) != NULL);
      }
   }

   view_current_image();
}

void
viewer_toolbar_rotate(GtkWidget *widget, gpointer data)
{
   gint i, j, bufsize, t, direction;
   guchar *newbuf, *ptr1, *ptr2;

   if (viewer_busy) return;

   if (!rc_get_boolean("full_screen"))
      viewer_save_enable(TRUE);

   direction = (gint) data;

   cache = scanline_get_cache();
   bufsize = cache->buffer_width * cache->buffer_height * 3;
   newbuf = g_malloc(bufsize);

   switch(direction)
   {
      case LEFT :   /* Rotate to left */
         for (i = 0; i < cache->buffer_height; i++)
         {
            for (j = 0; j < cache->buffer_width; j++)
            {
               ptr1 = newbuf + ((cache->buffer_width - j - 1)
                  * cache->buffer_height + i) * 3;
               ptr2 = cache->buffer +
                  (i * cache->buffer_width + j) * 3;
               *ptr1++ = *ptr2++;
               *ptr1++ = *ptr2++;
               *ptr1++ = *ptr2++;
            }
         }
         break;

      case RIGHT:   /* Rotate to right */
         for (i = 0; i < cache->buffer_height; i++)
         {
            for (j = 0; j < cache->buffer_width; j++)
            {
               ptr1 = newbuf + (j * cache->buffer_height +
                  (cache->buffer_height - i - 1)) * 3;
               ptr2 = cache->buffer +
                  (i * cache->buffer_width + j) * 3;
               *ptr1++ = *ptr2++;
               *ptr1++ = *ptr2++;
               *ptr1++ = *ptr2++;
            }
         }
         break;

      default:
         break;
   }

   g_free(cache->buffer);
   cache->buffer = newbuf;

   t                    = cache->buffer_width;
   cache->buffer_width  = cache->buffer_height;
   cache->buffer_height = t;

   /* Display the rotated image */
   gtk_preview_size(GTK_PREVIEW(viewport_image),
      cache->buffer_width, cache->buffer_height);

   gtk_widget_set_usize(viewport,
      min(cache->buffer_width, MAX_WIDTH),
      min(cache->buffer_height, MAX_HEIGHT));
/*
   gtk_widget_queue_resize(viewport_image);

   for (i = 0; i < cache->buffer_height; i++)
   {
      gtk_preview_draw_row(GTK_PREVIEW(viewport_image),
         cache->buffer + i * cache->buffer_width * 3,
         0, i, cache->buffer_width);
   }
   gtk_widget_set_usize(viewport,
      min(cache->buffer_width, MAX_WIDTH),
      min(cache->buffer_height, MAX_HEIGHT));
   gtk_widget_draw(viewport_image, NULL);

   if (GTK_IS_BUTTON(widget))
   {
      x_pos_button = widget->allocation.x + (widget->allocation.width >> 1);
      y_pos_button = widget->allocation.y + (widget->allocation.height>> 1);
      move_pointer = TRUE;
   }
*/
}

/* Code added by Jean P. Demailly */
void
viewer_toolbar_reflect(GtkWidget *widget, gpointer data)
{
   gint        i, j, bufsize, direction;
   guchar      *newbuf, *ptr1, *ptr2;

   if (viewer_busy) return;

   if (!rc_get_boolean("full_screen"))
      viewer_save_enable(TRUE);

   direction = (gint) data;

   cache    = scanline_get_cache();
   bufsize  = cache->buffer_width * cache->buffer_height * 3;
   newbuf   = g_malloc(bufsize);

   switch(direction)
   {
      case RIGHTLEFT:
         for (j = 0; j < cache->buffer_height; j++)
         {
            for (i = 0; i < cache->buffer_width; i++)
            {
               ptr1 = newbuf + (j * cache->buffer_width +
                        cache->buffer_width - i - 1) * 3;
               ptr2 = cache->buffer + (j * cache->buffer_width + i) * 3;
               *ptr1++ = *ptr2++;
               *ptr1++ = *ptr2++;
               *ptr1++ = *ptr2++;
            }
         }
         break;

      case UPDOWN:
         for (j = 0; j < cache->buffer_height; j++)
         {
            for (i = 0; i < cache->buffer_width; i++)
            {
               ptr1 = newbuf + ((cache->buffer_height - j - 1) *
                        cache->buffer_width + i) * 3;
               ptr2 = cache->buffer + (j * cache->buffer_width + i) * 3;
               *ptr1++ = *ptr2++;
               *ptr1++ = *ptr2++;
               *ptr1++ = *ptr2++;
            }
         }
         break;

      default:
         break;
   }

   g_free(cache->buffer);

   cache->buffer        = newbuf;

   /* Display the reflected image */
   gtk_preview_size(GTK_PREVIEW(viewport_image),
      cache->buffer_width, cache->buffer_height);

   gtk_widget_set_usize(viewport,
      min(cache->buffer_width, MAX_WIDTH),
      min(cache->buffer_height, MAX_HEIGHT));
/*
   gtk_widget_queue_resize(viewport_image);
   for (i = 0; i < cache->buffer_height; i++)
   {
      gtk_preview_draw_row(GTK_PREVIEW(viewport_image),
         cache->buffer + i * cache->buffer_width * 3,
         0, i, cache->buffer_width);
   }
   gtk_widget_draw(viewport_image, NULL);
*/
}

void
viewer_toolbar_refresh(GtkWidget *widget, gpointer data)
{
   viewer_save_enable(FALSE);
   view_current_image();
}

void
viewer_button_press(GtkWidget *widget, GdkEvent *event, gpointer data)
{
   static GtkWidget *label;

   if (viewer_busy) return;

   switch (event->key.keyval)
   {
      case GDK_Shift_L:
      case GDK_Shift_R:
         break;
/*
   menu = gtk_menu_new();

   menu_item = gtk_menu_item_new_with_label(_("Next image por las dudas lo"));
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_widget_show(menu_item);

   gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 0, 0);


         if (msg == NULL)
         {
            msg = gtk_window_new(GTK_WINDOW_POPUP);
            label = gtk_label_new("Mensaje");
            gtk_container_add(GTK_CONTAINER(msg), label);
            gtk_window_set_position(GTK_WINDOW(msg), GTK_WIN_POS_CENTER);

            gtk_widget_realize(msg);
            gdk_window_set_decorations(msg->window, 0);
            gdk_window_set_functions(msg->window, 0);

         }
         gtk_widget_show_all(msg);
*/
   }
}

void
viewer_key_release(GtkWidget *widget, GdkEvent *event, gpointer data)
{
   gboolean    fullscreen;

   if (viewer_busy) return;

   fullscreen = rc_get_boolean("full_screen");

   if (! fullscreen)
   {
      viewer_save_enable(FALSE);
   }

   switch (event->key.keyval)
   {
/*
      case GDK_Shift_L:
      case GDK_Shift_R:
         gtk_widget_hide(msg);
         break;
*/
      case 'f':
      case 'F':
         rc_set_boolean("fit_screen", !fullscreen);

         if (fullscreen)
            viewer_popup_view(NULL, NULL);
         else
            viewer_toolbar_full_screen(NULL, NULL);

         break;

      case 'l':
      case 'L':
         viewer_toolbar_rotate(NULL, (gint *) LEFT);
         break;

      case 'r':
      case 'R':
         viewer_toolbar_rotate(NULL, (gint *) RIGHT);
         break;

      case 's':
      case 'S':
         if (! fullscreen)
         {
            viewer_slideshow_pressed();
         } else
         {
            viewer_popup_slideshow(NULL, NULL);
         }

         break;

      case 'q':
      case 'Q':
         close_gtksee();
         break;

      case 'b':
      case 'B':
         if (! fullscreen)
         {
            if (GTK_WIDGET_VISIBLE(statusbar))
            {
               gtk_widget_hide(statusbar);
               rc_set_boolean("show_statusbar" , FALSE);
            } else
            {
               gtk_widget_show(statusbar);
               rc_set_boolean("show_statusbar", TRUE);
            }
         }
         view_current_image();

         break;

      case 't':
      case 'T':
         if (! fullscreen)
         {
            if (GTK_WIDGET_VISIBLE(toolbar))
            {
               gtk_widget_hide(toolbar);
               rc_set_boolean("show_toolbar", FALSE);
            } else
            {
               gtk_widget_show(toolbar);
               rc_set_boolean("show_toolbar", TRUE);
            }
         }
         break;

         /* Loop to the first if we are at the end of the list */
      case GDK_Home:
         if (imagelist == NULL)
         {
            image_infos = g_list_first(image_infos);
            current = (ImageInfo*) image_infos->data;
            if (!fullscreen)
            {
               viewer_next_enable(g_list_next(image_infos) != NULL);
               viewer_prev_enable(FALSE);
            }
         } else
         {
            current = image_list_get_first(IMAGE_LIST(imagelist));
            if (!fullscreen)
            {
               viewer_next_enable(max_image_list - current->serial);
               viewer_prev_enable(FALSE);
            }
         }
         view_current_image();
         break;

      case GDK_End:
         if (imagelist == NULL)
         {
            image_infos = g_list_last(image_infos);
            current = (ImageInfo*) image_infos->data;
            if (!fullscreen)
            {
               viewer_prev_enable(g_list_previous(image_infos) != NULL);
               viewer_next_enable(FALSE);
            }
         } else
         {
            current = image_list_get_last(IMAGE_LIST(imagelist));
            if (!fullscreen)
            {
               viewer_prev_enable(current->serial - min_image_list);
               viewer_next_enable(FALSE);
            }
         }
         view_current_image();
         break;

      case GDK_BackSpace:
      case GDK_Page_Up:
         if (fullscreen)
         {
            viewer_toolbar_prev_image(widget, NULL);
         } else
         {
            viewer_prev_pressed();
         }
         break;

      case GDK_space:
      case GDK_Page_Down:
         if (fullscreen)
         {
            viewer_toolbar_next_image(widget, NULL);
         } else
         {
            viewer_next_pressed();
         }

         break;
      case GDK_Escape:
         viewer_toolbar_browse(widget, NULL);
         break;

      case GDK_Left:
         dnd_viewport_move(DND_VIEWPORT(viewport), -MOVEMENT_INCREASE, 0);
         break;
      case GDK_Right:
         dnd_viewport_move(DND_VIEWPORT(viewport), MOVEMENT_INCREASE, 0);
         break;
      case GDK_Up:
         dnd_viewport_move(DND_VIEWPORT(viewport), 0, -MOVEMENT_INCREASE);
         break;
      case GDK_Down:
        dnd_viewport_move(DND_VIEWPORT(viewport), 0, MOVEMENT_INCREASE);
         break;
      default:
         break;
   }
}

/* Menu popup in fullscreen */
void
viewer_menu_popup(GtkWidget *widget, gpointer data)
{
   GtkWidget *menu, *menu_item;
   gboolean full_screen, fit_screen;

   if (viewer_busy) return;

   menu = gtk_menu_new();

   menu_item = gtk_menu_item_new();
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_widget_show(menu_item);

   menu_item = gtk_menu_item_new_with_label(_("Next image"));
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
      GTK_SIGNAL_FUNC(viewer_toolbar_next_image), NULL);
   if (imagelist == NULL)
   {
      gtk_widget_set_sensitive(
         menu_item,
         g_list_next(image_infos) != NULL);
   } else
   {
      gtk_widget_set_sensitive(
         menu_item,
         image_list_get_next(IMAGE_LIST(imagelist), current) != NULL);
   }
   gtk_widget_show(menu_item);

   menu_item = gtk_menu_item_new_with_label(_("Previous image"));
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
      GTK_SIGNAL_FUNC(viewer_toolbar_prev_image), NULL);
   if (imagelist == NULL)
   {
      gtk_widget_set_sensitive(
         menu_item,
         g_list_previous(image_infos) != NULL);
   } else
   {
      gtk_widget_set_sensitive(
         menu_item,
         image_list_get_previous(IMAGE_LIST(imagelist), current) != NULL);
   }
   gtk_widget_show(menu_item);

   sshow_item = gtk_check_menu_item_new_with_label(_("Slide show"));
   gtk_menu_append(GTK_MENU(menu), sshow_item);

   gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(sshow_item), slide_show);
   gtk_signal_connect(GTK_OBJECT(sshow_item), "toggled",
         GTK_SIGNAL_FUNC(viewer_popup_slideshow), NULL);
   gtk_widget_show(sshow_item);

   menu_item = gtk_menu_item_new();
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_widget_show(menu_item);

   menu_item = gtk_check_menu_item_new_with_label(_("Fit screen"));
   gtk_menu_append(GTK_MENU(menu), menu_item);

   fit_screen = rc_get_boolean("fit_screen");

   gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(menu_item), fit_screen);
   gtk_signal_connect(GTK_OBJECT(menu_item), "toggled",
         GTK_SIGNAL_FUNC(viewer_popup_fitscreen), NULL);

   gtk_widget_show(menu_item);

   menu_item = gtk_check_menu_item_new_with_label(_("Full screen"));
   gtk_menu_append(GTK_MENU(menu), menu_item);

   full_screen = rc_get_boolean("full_screen");
   if (full_screen)
   {
      gtk_check_menu_item_set_state(
         GTK_CHECK_MENU_ITEM(menu_item),
         TRUE);
      gtk_signal_connect(GTK_OBJECT(menu_item), "toggled",
         GTK_SIGNAL_FUNC(viewer_popup_view), NULL);
   } else
   {
      gtk_check_menu_item_set_state(
         GTK_CHECK_MENU_ITEM(menu_item),
         FALSE);
      gtk_signal_connect(GTK_OBJECT(menu_item), "toggled",
         GTK_SIGNAL_FUNC(viewer_toolbar_full_screen), NULL);
   }
   gtk_widget_show(menu_item);

   menu_item = gtk_menu_item_new();
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_widget_show(menu_item);

   menu_item = gtk_menu_item_new_with_label(_("Browser"));
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
      GTK_SIGNAL_FUNC(viewer_toolbar_browse), NULL);
   gtk_widget_show(menu_item);

   menu_item = gtk_menu_item_new();
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_widget_show(menu_item);

   /* Rotate and reflect function ... */
   menu_item = gtk_menu_item_new_with_label(_("Rotate -90"));
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
      GTK_SIGNAL_FUNC(viewer_toolbar_rotate), (gint *) LEFT);
   gtk_widget_show(menu_item);

   menu_item = gtk_menu_item_new_with_label(_("Rotate +90"));
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
      GTK_SIGNAL_FUNC(viewer_toolbar_rotate), (gint *) RIGHT);
   gtk_widget_show(menu_item);

   menu_item = gtk_menu_item_new_with_label(_("Reflect Right/Left"));
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
      GTK_SIGNAL_FUNC(viewer_toolbar_reflect),
      (gint *) RIGHTLEFT);
   gtk_widget_show(menu_item);

   menu_item = gtk_menu_item_new_with_label(_("Reflect Up/Down"));
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
      GTK_SIGNAL_FUNC(viewer_toolbar_reflect),
      (gint *) UPDOWN);
   gtk_widget_show(menu_item);

   menu_item = gtk_menu_item_new();
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_widget_show(menu_item);

   menu_item = gtk_menu_item_new_with_label(_("Exit"));
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
      GTK_SIGNAL_FUNC(close_gtksee), NULL);
   gtk_widget_show(menu_item);

   gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 3, 1);
}

void
viewer_popup_view(GtkWidget *widget, gpointer data)
{
   if (viewer_busy) return;

   rc_set_boolean("full_screen", FALSE);

   gtk_widget_destroy(window);
   get_viewer_window_internal();
}

void
viewer_popup_slideshow(GtkWidget *widget, gpointer data)
{
   if (viewer_busy) return;

   viewer_slideshow_toggled_internal(!slide_show);

}

void
viewer_popup_fitscreen(GtkWidget *widget, gpointer data)
{
   if (viewer_busy) return;

   rc_set_boolean("fit_screen", GTK_CHECK_MENU_ITEM(widget)->active);

   view_current_image();
}

/* Functions called from gtksee.c */
GtkWidget*
get_viewer_window(GtkWidget *il)
{
   gboolean full_screen;

   /* setting pointers and values... */
   imagelist   = il;
   image_infos = NULL;

   if (viewer_watch_cursor == NULL)
   {
      viewer_watch_cursor = gdk_cursor_new(GDK_WATCH);
   }

   viewer_busy = FALSE;

   /* view the selected image if selected,
    * otherwise, view the first.
    */
   current = image_list_get_selected(IMAGE_LIST(imagelist));
   if (current == NULL || !current->valid || current->width < 0)
      current = image_list_get_first(IMAGE_LIST(imagelist));

   min_image_list = image_list_get_first(IMAGE_LIST(imagelist))->serial;
   max_image_list = image_list_get_last (IMAGE_LIST(imagelist))->serial;

   full_screen = rc_get_boolean("full_screen");
   have_xgrab = FALSE;
   if (full_screen)
   {
      return get_full_view_window();
   } else
   {
      return get_viewer_window_internal();
   }
}

GtkWidget*
get_viewer_window_with_files(GList *infos, gint max_files)
{
   gboolean full_screen;

   /* setting pointers and values... */
   imagelist   = NULL;
   image_infos = g_list_first(infos);
   if (viewer_watch_cursor == NULL)
   {
      viewer_watch_cursor = gdk_cursor_new(GDK_WATCH);
   }

   viewer_busy = FALSE;

   /* view the selected image if selected,
    * otherwise, view the first.
    */
   current = (ImageInfo*) image_infos->data;
   if (current == NULL || !current->valid || current->width < 0)
   {
      gchar *root;

      root  = gtksee_malloc(BUF_STD);
      strncpy(root, "/", 2);
      gtksee_main(root);
   }

   min_image_list = 1;
   max_image_list = max_files;

   full_screen = rc_get_boolean("full_screen");
   have_xgrab = FALSE;
   if (full_screen)
   {
      return get_full_view_window();
   } else
   {
      return get_viewer_window_internal();
   }
}

GtkWidget*
get_viewer_window_internal()
{
   GtkWidget    *table;
   GdkPixmap    *pixmap;
   GdkBitmap    *mask;
   GtkStyle     *style;
   GdkEventMask event_mask;

   /* creating viewer window */
   window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_signal_connect (GTK_OBJECT(window), "delete_event",
      GTK_SIGNAL_FUNC(close_gtksee), NULL);
   gtk_container_border_width(GTK_CONTAINER(window), 2);
   gtk_window_set_policy(GTK_WINDOW(window), FALSE, TRUE, TRUE);

   gtk_window_set_title(GTK_WINDOW(window), "GTK See");
   gtk_window_set_wmclass(GTK_WINDOW(window), "viewer", "GTKSee");
   event_mask = gtk_widget_get_events(window);
   event_mask |= GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK;
   gtk_widget_set_events(window, event_mask);

   gtk_signal_connect(
      GTK_OBJECT(window),
      "key_release_event",
      GTK_SIGNAL_FUNC(viewer_key_release),
      NULL);
/*
   gtk_signal_connect(
      GTK_OBJECT(window),
      "size-allocate",
      GTK_SIGNAL_FUNC(viewer_move_pointer),
      NULL);
*/
/*
   gtk_signal_connect_after(
      GTK_OBJECT(window),
      "size-request",
      GTK_SIGNAL_FUNC(viewer_move_pointer),
      NULL);
*/

   if (rc_get_boolean("center_window"))
#if ((GTK_MAJOR_VERSION>1) || \
     (GTK_MAJOR_VERSION==1 && GTK_MINOR_VERSION>2) || \
     (GTK_MAJOR_VERSION==1 && GTK_MINOR_VERSION==2 && GTK_MICRO_VERSION>=5))
   gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER_ALWAYS);
#else
   gtk_widget_set_uposition(window, 0, 0);
#endif


   gtk_widget_realize(window);

   old_style = style = gtk_widget_get_style(window);

   pixmap = gdk_pixmap_create_from_xpm_d(
      window->window,
      &mask, &style->bg[GTK_STATE_NORMAL],
      (gchar **)gtksee_xpm);
   gdk_window_set_icon(window->window, NULL, pixmap, mask);
   gdk_window_set_icon_name(window->window, "GTK See Icon");
   g_free(pixmap);
   g_free(mask);

   /* disable toolbar events. */
   viewer_busy = 1;

   /* creating toolbar */
   toolbar = get_viewer_toolbar(window);
   viewer_slideshow_set_state(slide_show);
   gtk_widget_show(toolbar);

   /* creating viewer statusbar */
   pixmap = gdk_pixmap_create_from_xpm_d(
      window->window,
      &mask, &style->bg[GTK_STATE_NORMAL],
      (gchar **)blank_xpm);
   statusbar = get_viewer_status_bar(pixmap, mask);
   gtk_widget_show(statusbar);

   viewer_save_enable(FALSE);

   viewer_busy = 0;

   /* creating frame */
   frame = gtk_frame_new(NULL);
   gtk_widget_show(frame);

   viewport = dnd_viewport_new();

/* OJO, ES UTIL PARA TRABAJO POSTERIOR
   gtk_signal_connect(
      GTK_OBJECT(viewport),
      "popup",
      GTK_SIGNAL_FUNC(viewer_menu_popup),
      NULL);
*/
   gtk_signal_connect(
      GTK_OBJECT(viewport),
      "double_clicked",
      GTK_SIGNAL_FUNC(viewer_toolbar_browse),
      NULL);

   gtk_signal_connect(
      GTK_OBJECT(viewport),
      "wheel_down",
      GTK_SIGNAL_FUNC(viewer_toolbar_next_image),
      NULL);

   gtk_signal_connect(
      GTK_OBJECT(viewport),
      "wheel_up",
      GTK_SIGNAL_FUNC(viewer_toolbar_prev_image),
      NULL);

   gtk_container_add(GTK_CONTAINER(frame), viewport);
   gtk_widget_show(viewport);

   viewport_image = gtk_preview_new(GTK_PREVIEW_COLOR);

   gtk_container_add(GTK_CONTAINER(viewport), viewport_image);
   gtk_widget_show(viewport_image);

   gtk_signal_connect(
      GTK_OBJECT(viewport_image),
      "expose_event",
      GTK_SIGNAL_FUNC(repaint_viewport_image),
      NULL);

   /* creating viewer table */
   table = gtk_table_new(3, 1, FALSE);
   gtk_widget_show(table);

   /* adding toolbar, frame & statusbar into table */
   gtk_table_attach(GTK_TABLE(table), toolbar, 0, 1, 0, 1, GTK_FILL|GTK_EXPAND, 0, 0, 2);
   gtk_table_attach_defaults(GTK_TABLE(table), frame, 0, 1, 1, 2);
   gtk_table_attach(GTK_TABLE(table), statusbar, 0, 1, 2, 3, GTK_FILL|GTK_EXPAND, 0, 0, 2);

   /* adding table into viewer window */
   gtk_container_add(GTK_CONTAINER(window), table);
   gtk_widget_show(window);

   if (imagelist == NULL)
   {
      image_list_load_pixmaps(window);
      viewer_next_enable(g_list_next(image_infos) != NULL);
      viewer_prev_enable(g_list_previous(image_infos) != NULL);
   } else
   {
      if (min_image_list == current->serial) viewer_prev_enable(FALSE);
      if (max_image_list == current->serial) viewer_next_enable(FALSE);
   }

   /* Create new style for background color */
   if (rc_get_int("background_color") != -1)
   {
      GdkColor backcolor;

      backcolor = gint_to_gdkcolor(rc_get_int("background_color"));
      style = gtk_style_new();
      style->bg[GTK_STATE_NORMAL] = backcolor;
      gtk_widget_set_style(viewport, style);
      scanline_init(gdk_color_context_new(
                     gdk_window_get_visual(window->window),
                     gdk_window_get_colormap(window->window)),
                     style);
   }

   if (rc_get_boolean("show_toolbar") == FALSE)
   {
      gtk_widget_hide(toolbar);
   }

   if (rc_get_boolean("show_statusbar") == FALSE)
   {
      gtk_widget_hide(statusbar);
   }

   view_current_image();

   always_on_top(rc_get_boolean("top_window"));

   return window;
}

GtkWidget*
get_full_view_window()
{
   GdkEventMask event_mask;
   GtkStyle     *style;

   window = gtk_window_new(GTK_WINDOW_POPUP);

   gtk_container_border_width(GTK_CONTAINER(window), 0);
   gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);
   gtk_widget_set_usize(window, MAX_FULL_WIDTH, MAX_FULL_HEIGHT);
   gtk_widget_set_uposition(window, 0, 0);

   event_mask = gtk_widget_get_events(window);
   event_mask |= GDK_KEY_RELEASE_MASK;
   gtk_widget_set_events(window, event_mask);
   gtk_signal_connect(
      GTK_OBJECT(window),
      "key_release_event",
      GTK_SIGNAL_FUNC(viewer_key_release),
      NULL);
/*
   gtk_signal_connect(
      GTK_OBJECT(window),
      "key_press_event",
      GTK_SIGNAL_FUNC(viewer_key_press),
      NULL);
*/
   old_style =  style = gtk_widget_get_style(window);

   /* cheating frame */
   frame = window;

   viewport = dnd_viewport_new();
   gtk_signal_connect(
      GTK_OBJECT(viewport),
      "double_clicked",
      GTK_SIGNAL_FUNC(viewer_toolbar_browse),
      NULL);

   /* use popup menu only in full-screen mode */
   gtk_signal_connect(
      GTK_OBJECT(viewport),
      "popup",
      GTK_SIGNAL_FUNC(viewer_menu_popup),
      NULL);

   gtk_signal_connect(
      GTK_OBJECT(viewport),
      "wheel_down",
      GTK_SIGNAL_FUNC(viewer_toolbar_next_image),
      NULL);

   gtk_signal_connect(
      GTK_OBJECT(viewport),
      "wheel_up",
      GTK_SIGNAL_FUNC(viewer_toolbar_prev_image),
      NULL);

   gtk_container_add(GTK_CONTAINER(frame), viewport);
   gtk_widget_show(viewport);

   viewport_image = gtk_preview_new(GTK_PREVIEW_COLOR);

   gtk_signal_connect(
      GTK_OBJECT(viewport_image),
      "expose_event",
      GTK_SIGNAL_FUNC(repaint_viewport_image),
      NULL);

   gtk_container_add(GTK_CONTAINER(viewport), viewport_image);
   gtk_widget_show(viewport_image);

   gtk_widget_realize(window);
   gdk_window_set_decorations(window->window, 0);
   gdk_window_set_functions(window->window, 0);
   gtk_widget_show(window);

   if (imagelist == NULL)
   {
      image_list_load_pixmaps(window);
   }

   /* Create new style for background color */
   if (rc_get_int("background_color") != -1)
   {
      GdkColor backcolor;

      backcolor = gint_to_gdkcolor(rc_get_int("background_color"));
      style = gtk_style_new();
      style->bg[GTK_STATE_NORMAL] = backcolor;
      gtk_widget_set_style(viewport, style);
      scanline_init(gdk_color_context_new(
                     gdk_window_get_visual(window->window),
                     gdk_window_get_colormap(window->window)),
                     style);
   }

   view_current_image();

   /* Make X grab to full-screen window */
   if ((gdk_pointer_grab(window->window, TRUE,
            GDK_BUTTON_PRESS_MASK   |
            GDK_BUTTON_RELEASE_MASK |
            GDK_ENTER_NOTIFY_MASK   |
            GDK_LEAVE_NOTIFY_MASK   |
            GDK_POINTER_MOTION_MASK,
            NULL, NULL, 0) == 0))
   {
      if (gdk_keyboard_grab(window->window, TRUE, 0) != 0)
      {
         gdk_pointer_ungrab(0);
      } else
      {
         have_xgrab = TRUE;
      }
   }

   return window;
}

