/*
 * GTK See -- an image viewer based on GTK+
 * Copyright (C) 1998 Hotaru Lee <jkhotaru@mail.sti.com.cn> <hotaru@163.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Many codes and ideas are taken from:
 *  + ICON plug-in for the GIMP - Copyright (C) 2002 Christian Kreibich
 *  + Icoconvert - Maintained by Chris Moates
 *
 * And transparent support for 16-color icon was added
 *
 * 2004-10-31: Added support for 1, 8, 24 and 32 bits Windows .ICO
 *             Added support for Windows .CUR files
 *
 * 2004-11-08: Fixed bug showing 24-bits ico
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include "im_ico.h"

typedef struct {
   gshort idReserved;  /* always set to 0 */
   gshort idType;      /* always set to 1 */
   gshort idCount;     /* number of icon images */
   /* immediately followed by idCount TIconDirEntries */
} ICO_Header;

typedef struct {
   guchar  bWidth;         /* Width */
   guchar  bHeight;        /* Height */
   guint   bColorCount;    /* # of colors used, see below */
   guchar  bReserved;      /* not used, 0 */
   gushort wPlanes;        /* not used, 0 */
   gushort wBitCount;      /* not used, 0 */
   gulong  dwBytesInRes;   /* total number of bytes in image */
   gulong  dwImageOffset;  /* location of image from the beginning of file */
} ICO_IconDirEntry;

typedef struct {
   glong  biSize;          /* sizeof(TBitmapInfoHeader) */
   glong  biWidth;         /* width of bitmap */
   glong  biHeight;        /* height of bitmap, see notes */
   gshort biPlanes;        /* planes, always 1 */
   gshort biBitCount;      /* number of color bits */
   glong  biCompression;   /* compression used, 0 */
   glong  biSizeImage;     /* size of the pixel data, see notes */
   glong  biXPelsPerMeter; /* not used, 0 */
   glong  biYPelsPerMeter; /* not used, 0 */
   glong  biClrUsed;       /* # of colors used, set to 0 */
   glong  biClrImportant;  /* important colors, set to 0 */
} ICO_BitmapInfoHeader;

typedef struct {
   ICO_BitmapInfoHeader icHeader;   /* image header info */
   guchar               *icColors;  /* image palette */
} ICO_IconImage;

gboolean ICO_read_iconheader  (FILE *, ICO_Header *);
gboolean ICO_read_direntry    (FILE *, ICO_IconDirEntry *);
gboolean ICO_read_bitmapinfo  (FILE *, ICO_IconImage *, ICO_IconDirEntry *);
gint     length_map           (gint , gint , gint);
gint     get_bit_from_AND     (guchar *, gint, gint, gint);

/* Begin the code... */
gboolean
ico_get_header (gchar *filename, ico_info *info)
{
   FILE              *fp;
   ICO_Header        ICO_header;
   ICO_IconDirEntry  direntry, tmp;
   ICO_IconImage     iconimage;
   guint             i, index=0;

   fp = fopen(filename, "rb");
   if (fp == NULL) return FALSE;

   if (ICO_read_iconheader(fp, &ICO_header) == FALSE)
   {
      fclose(fp);
      return FALSE;
   }

   /* Search the biggest icon */
   for (i=0; i<ICO_header.idCount; i++)
   {
      gint score, color;

      if (ICO_read_direntry(fp, &tmp) == FALSE)
      {
         fclose(fp);
         return FALSE;
      }

      color = (tmp.bColorCount!=0) ? tmp.bColorCount : 256;
      score = color*1024 + tmp.bWidth*tmp.bHeight;
      if (score > index)
      {
         index    = score;
         direntry = tmp;
      }
   }

   fseek(fp, direntry.dwImageOffset, SEEK_SET);
   if (ICO_read_bitmapinfo(fp, &iconimage, &direntry) == FALSE)
   {
      fclose(fp);
      return FALSE;
   }

   info -> width = (gint) direntry.bWidth;
   info -> height= (gint) direntry.bHeight;

   if (direntry.bColorCount == 0)
   {
      info -> ncolors = iconimage.icHeader.biBitCount;
   } else
   {
      info -> ncolors = 0;
      for (i = direntry.bColorCount; i > 1; i >>= 1, info->ncolors++) ;
   }

   fclose(fp);
   return TRUE;
}

gboolean
ico_load (gchar *filename, IcoLoadFunc func)
{
   FILE              *fp;
   ICO_Header        ICO_header;
   ICO_IconDirEntry  direntry, tmp;
   ICO_IconImage     iconimage;
   gint              x, y, line, offset, width32;
   gint              i, width, height;
   guint             transparent;
   guint             *buf, pos, index=0;
   guchar            *icXOR, *icAND, *icPalette[3], bitAND;
   gushort           bpp;
   gint              length;

   fp = fopen(filename, "rb");
   if (fp == NULL) return FALSE;

   if (ICO_read_iconheader(fp, &ICO_header) == FALSE)
   {
      fclose(fp);
      return FALSE;
   }

   /* Search the biggest icon */
   for (i=0; i<ICO_header.idCount; i++)
   {
      gint score;
      gint color;

      if (ICO_read_direntry(fp, &tmp) == FALSE)
      {
         fclose(fp);
         return FALSE;
      }

      color = (tmp.bColorCount) ? tmp.bColorCount : 256;
      score = color*1024 + tmp.bWidth*tmp.bHeight;

      if (score > index)
      {
         index   = score;
         direntry= tmp;
      }
   }

   fseek(fp, direntry.dwImageOffset, SEEK_SET);

   if (ICO_read_bitmapinfo(fp, &iconimage, &direntry) == FALSE)
   {
      fclose(fp);
      return FALSE;
   }

   width   = iconimage.icHeader.biWidth;
   height  = iconimage.icHeader.biHeight / 2;
   bpp     = iconimage.icHeader.biBitCount;
   icXOR   = NULL;
   icAND   = NULL;

   if (direntry.bColorCount == 0)
   {
      if (iconimage.icHeader.biBitCount >= 8)
         transparent = 256;
      else
         transparent = 1 << (iconimage.icHeader.biBitCount);
   } else
   {
      transparent = direntry.bColorCount;
   }

   if (bpp <= 8)
   {
      gint clrused = iconimage.icHeader.biClrUsed;
      guchar c;


      if (clrused == 0) clrused = 1 << bpp;

      icPalette[0] = g_malloc(sizeof(guchar) * clrused);
      icPalette[1] = g_malloc(sizeof(guchar) * clrused);
      icPalette[2] = g_malloc(sizeof(guchar) * clrused);

      for (i=0; i<clrused; i++)
      {
         fread((void *)&c, sizeof(c), 1, fp);
         icPalette[2][i] = c;

         fread((void *)&c, sizeof(c), 1, fp);
         icPalette[1][i] = c;

         fread((void *)&c, sizeof(c), 1, fp);
         icPalette[0][i] = c;

         fread((void *)&c, sizeof(c), 1, fp);
      }
   } else
   {
      icPalette[0] = NULL;
   }

   /* Read the XOR map */
   length= length_map(width, height, bpp);
   icXOR = g_malloc0(sizeof(guchar) * length);
   fread((void *)icXOR, sizeof(guchar), length, fp);

   /* Read the AND map */
   length= length_map(width, height, 1);
   icAND = g_malloc0(sizeof(guchar) * length);
   fread((void *)icAND, 1, length, fp);

   /*
   printf("length XOR: %i\n", length);
   printf("length AND: %i\n", length);
   printf("width     : %i\n", width);
   printf("height    : %i\n", height);
   printf("bpp       : %i\n", bpp);
   dump_andmask(&iconimage, icAND);
   */

   fclose(fp);

   buf = g_malloc0(sizeof(guint) * width * 4 * bpp * height);

   for (y=0; y<height; y++)
   {
      i = 0;
      for (x=0; x<width; x++)
      {
         index  = y * width + x;
         bitAND = get_bit_from_AND(icAND, width, index, x);

         switch(bpp)
         {
            /* 2-bits ICO */
            case 1:
               width32= (width % 32 == 0) ? width/32 : width/32 + 1;
               line   = index / width;
               offset = index % width;
               pos    = icXOR[line * width32 * 4 + offset/8] &
                           (1<<(7- x % 8));
               buf[i++] = (bitAND) ? transparent : ((pos == 0) ? 0 : 1);

               break;

            /* 4-bits ICO */
            case 4:
               width32= (width % 8 == 0) ? width/8 : width/8+1;
               line   = index / width;
               offset = index % width;
               pos    = (icXOR[line * width32 * 4 + offset/2] &
                           (0x0F << (4 * (1 - index % 2))));
               if (index % 2 == 0) pos = pos >> 4;

               buf[x] = (bitAND) ? transparent : pos;

               break;

            /* 8-bits ICO */
            case 8:
               width32= (width % 4 == 0) ? width/4 : width/4+1;
               line   = index / width;
               offset = index % width;
               pos    = icXOR[line * width32 * 4 + offset];

               buf[x] = (bitAND) ? transparent : pos;

               break;

            /* 24-bits 32-bits ICO */
            case 24:
            case 32:
               width32= ((width * bpp)% 32 == 0) ?
                           (width*bpp)/32        :
                           (width*bpp)/32 + 1;
               line   = index / width;
               offset = index % width;

               pos = line * width32 * 4 + offset * (bpp / 8);

               buf[i+0]= (bitAND) ? transparent : icXOR[pos+2];
               buf[i+1]= (bitAND) ? transparent : icXOR[pos+1];
               buf[i+2]= (bitAND) ? transparent : icXOR[pos+0];
               i += 3;

               break;

            default:
               break;
         }
      }

      if ((*func) (buf, icPalette, transparent, width, height-y-1, 0)) break;
   }
   /* Finish... */

   g_free(buf);
   g_free(icXOR);
   g_free(icAND);
   if (icPalette[0] != NULL)
   {
      g_free(icPalette[0]);
      g_free(icPalette[1]);
      g_free(icPalette[2]);
   }

   return TRUE;
}

gboolean
ICO_read_iconheader(FILE *fp, ICO_Header *header)
{
   gushort i;

   /* read Reserved bit */
   if (fread((void *)&i, 1, sizeof(i), fp) != sizeof(i)) return FALSE;
   header -> idReserved = i;

   /* read Resource Type, 1 is for icons, abort if different */
   if (fread((void *)&i,1,sizeof(i),fp) != sizeof(i)) return FALSE;
   header -> idType = i;
   if (header -> idType != 1 && header -> idType != 2) return FALSE;

   /* read Number of images, abort if invalid value (0) */
   if (fread((void *)&i,1,sizeof(i),fp) != sizeof(i)) return FALSE;
   header -> idCount = i;              /* Number of images (>0) */
   if (header -> idCount==0) return FALSE;

   return TRUE;
}

gboolean
ICO_read_direntry(FILE *fp, ICO_IconDirEntry *direntry)
{
   gushort i;
   guchar c;
   gulong l;

   /* read Width, in pixels */
   if (fread((void *)&c,1,sizeof(c),fp) != sizeof(c)) return FALSE;
   direntry -> bWidth = c;

   /* read Height, in pixels */
   if (fread((void *)&c,1,sizeof(c),fp) != sizeof(c)) return FALSE;
   direntry -> bHeight = c;

   if (direntry -> bWidth < 1 || direntry -> bHeight < 1) return FALSE;

   /* Number of colors in image */
   if (fread((void *)&c,1,sizeof(c),fp) != sizeof(c)) return FALSE;
   direntry -> bColorCount = c;

   if (direntry -> bColorCount>256) return FALSE;

   /* Reserved (must be 0) */
   if (fread((void *)&c,1,sizeof(c),fp) != sizeof(c)) return FALSE;
   direntry -> bReserved = c;

   /* Color Planes, not used, 0 */
   if (fread((void *)&i,1,sizeof(i),fp) != sizeof(i)) return FALSE;
   direntry -> wPlanes = i;

   /* Bits per pixel, not used, 0 */
   if (fread((void *)&i,1,sizeof(i),fp) != sizeof(i)) return FALSE;
   direntry -> wBitCount = i;

   /* size of image data, in bytes */
   if (fread((void *)&l,1,sizeof(l),fp) != sizeof(l)) return FALSE;
   direntry -> dwBytesInRes = l;
   if (direntry -> dwBytesInRes == 0) return FALSE;

   /* offset from beginning image info */
   if (fread((void *)&l,1,sizeof(l),fp) != sizeof(l)) return FALSE;
   direntry -> dwImageOffset = l;
   if (direntry -> dwImageOffset == 0) return FALSE;

   return TRUE;
}

gboolean
ICO_read_bitmapinfo(FILE *fp, ICO_IconImage *iconimage, ICO_IconDirEntry *direntry)
{
   gulong l;
   gushort i;

   /* read bitmap info an perform some primitive sanity checks */

   /* sizeof(TBitmapInfoHeader) */
   if (fread((void *)&l,1,sizeof(l),fp) != sizeof(l)) return FALSE;
   iconimage -> icHeader.biSize = l;

   /* width of bitmap */
   if (fread((void *)&l,1,sizeof(l),fp) != sizeof(l)) return FALSE;
   iconimage -> icHeader.biWidth = l;

   /* height of bitmap, see notes (icon.h) */
   if (fread((void *)&l,1,sizeof(l),fp) != sizeof(l)) return FALSE;
   iconimage -> icHeader.biHeight = l;

   if (iconimage -> icHeader.biWidth < 1 || iconimage -> icHeader.biHeight < 1)
      return FALSE;

   /* planes, always 1 */
   if (fread((void *)&i,1,sizeof(i),fp) != sizeof(i)) return FALSE;
   iconimage -> icHeader.biPlanes = i;

   /* number of color bits (1, 4, 8, 24, 32) */
   if (fread((void *)&i,1,sizeof(i),fp) != sizeof(i)) return FALSE;
   /* Only 1, 4, 8, 24 or 32 bits support... */
   if (i!=1 && i!=4 && i!=8 && i!=24 && i!=32) return FALSE;
   iconimage -> icHeader.biBitCount = i;

   /* compression used, 0 */
   if (fread((void *)&l,1,sizeof(l),fp) != sizeof(l)) return FALSE;
   iconimage -> icHeader.biCompression = l;
   if (iconimage -> icHeader.biCompression != 0) return FALSE;

   /* size of the pixel data, see icon.h */
   if (fread((void *)&l,1,sizeof(l),fp) != sizeof(l)) return FALSE;
   iconimage -> icHeader.biSizeImage = l;

  /* used to abort on this, but it's not really that important...
      should be the len of the XORMASK + ANDMASK...

      as the saying goes, be liberal in what you accept,
      and strict in what you output (or something to that effect
    */

/*  if (ih.biSizeImage==0)
    return(-1); */

   /* not used, 0 */
   if (fread((void *)&l,1,sizeof(l),fp) != sizeof(l)) return FALSE;
   iconimage -> icHeader.biXPelsPerMeter = l;

   /* not used, 0 */
   if (fread((void *)&l,1,sizeof(l),fp) != sizeof(l)) return FALSE;
   iconimage -> icHeader.biYPelsPerMeter = l;

   /* # of colors used, set to 0 */
   if (fread((void *)&l,1,sizeof(l),fp) != sizeof(l)) return FALSE;
   iconimage -> icHeader.biClrUsed = l;

   /* important colors, set to 0 */
   if (fread((void *)&l,1,sizeof(l),fp) != sizeof(l)) return FALSE;
   iconimage -> icHeader.biClrImportant = l;

   return TRUE;
}

gint
dump_andmask(ICO_IconImage *ii, guchar *icAND)
{
   register int i, j, k;
   int base = 0;
   int height, padwidth;
   height=ii->icHeader.biHeight/2;
   padwidth=ii->icHeader.biWidth+ii->icHeader.biWidth%32;

   for(i = height-1; i >= 0; --i)
   {
      for(j = 0; j < ii->icHeader.biWidth; j += 8)
      {
         for(k = 7; k >= 0; --k)
         {
            base = *(icAND + ((i * padwidth + j) / 8)) &(1 << k);
            printf("%c", base ? '_' : 'X');
         }
      }
      fputc('\n', stdout);
   }
   fputc('\n', stdout);
   printf("Padwidth: %i\n", padwidth);
   return 0;
}

gint
length_map(gint width, gint height, gint bpp)
{
   gint len;

   switch (bpp)
   {
      case 1:
         if ((width % 32) == 0)
            len = (width * height / 8);
         else
            len = 4 * ((width/32 + 1) * height);
         break;

      case 4:
         if ((width % 8) == 0)
            len = (width * height / 2);
         else
            len = 4 * ((width/8 + 1) * height);
         break;

      case 8:
         if ((width % 4) == 0)
            len = width * height;
         else
            len = 4 * ((width/4 + 1) * height);
         break;

      default:
         if ((width % 32) == 0)
            len = width * height * (bpp/8);
         else
            len = 4 * (((width*bpp)/32 + 1) * height);
         break;
   }

   return len;
}

gint
get_bit_from_AND(guchar *icAND, gint width, gint bit, gint xactual)
{
   gint line, width32, offset, result;

   width32= (width % 32 == 0) ? width/32 : width/32 + 1;
   line   = bit / width;
   offset = bit % width;
   result = icAND[line * width32 * 4 + offset/8] & (1<<(7 - xactual %8));

   return (result ? 1 : 0);
}
