# -*- coding: utf-8 -*-
import gtk
from gtk import gdk
from gtk import keysyms
import pango
import cairo
import gobject
import locale
import math

SCALE = 1000.
FONT = "Sans 24"
zh_CN_FONT = "文鼎PL中楷Uni 24"
FONT_NUMBER = "Sans 24"
COLOR_CROSS     = (0, 0, 0, 0.2)
COLOR_HIGHLIGHT = (0, 0, 1, 1.0)
COLOR_CHAR      = (0, 0, 0, 0.2)
COLOR_NUMBER    = (0, 0, 1.0, 1.0)
CANVAS_NORMAL   = 0
CANVAS_EDITING  = 1

class Canvas (gtk.DrawingArea):
	
	__gsignals__ = {
	'strokes-changed': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE,
				())
	}

	def __init__ (self, config):
		gtk.DrawingArea.__init__ (self)
		self.connect ("expose-event", self.__expose_cb)
		self.connect ("button-press-event", self.__button_press_cb)
		self.connect ("key-press-event", self.__key_press_cb)
		self.connect ("size-allocate", self.__size_allocate_cb)
		self.add_events (gdk.BUTTON_PRESS_MASK)
		self.set_flags (gtk.CAN_FOCUS)
		self._strokes = []
		self._curr_stroke = []
		char_font = config.get("font", "char_font")
		if char_font == None:
			if locale.getlocale()[0] == "zh_CN":
				char_font = zh_CN_FONT
			else:
				char_font = FONT
		self.set_font(char_font)
		self._font_desc_number = pango.FontDescription (FONT_NUMBER)
		self._state = CANVAS_NORMAL
		self._chars = u" "
		self.set_size_request (400, 400)
		self._selection = []
		self._modified = False

	def set_font (self, font):
		"""set_font (self, font)
		font: font name, example "AR PL ZenKai Uni 27"
		"""
		self._font_desc = pango.FontDescription (font)
		self.queue_draw ()
	def is_modified (self):
		return self._modified
	
	def get_font (self):
		return self._font_desc.to_string ()

	def set_chars (self, chars, strokes = [], modified = False):
		"""set_chars (self, chars, strokes = [])
		chars: is an unicode chars
		strokes: a list for stroke, default = [] empty
		"""
		self._chars = chars
		self._strokes = strokes
		self._modified = False
		self.queue_draw ()
		self.emit ("strokes-changed")

	def get_strokes (self):
		"""get_strokes(self)
		return strokes list. [[(x1,y1),(x2,y2)],[(x3,y3),(x4,y4)]]
		"""
		return self._strokes[:]

	def set_strokes (self, strokes, modified = False):
		self._strokes = strokes
		self.queue_draw ()
		self.emit ("strokes-changed")
		self._modified = modified
	
	def delete_stroke (self, i = -1):
		"""delete_stroke (self, i = -1)
		Remove ith stroke, if i == -1, it will remove the last stroke. 
		"""
		if i == -1:
			if self._curr_stroke:
				self._curr_stroke = []
			elif self._strokes:
				self._strokes.pop ()
				self.emit ("strokes-changed")
		else:			
			del self._strokes[i]
			self.emit ("strokes-changed")
		self._modified = True
		self.queue_draw ()

	def delete_strokes (self, strokes):
		tmp = strokes[:]
		tmp.sort(reverse=True)
		for i in tmp:
			self.delete_stroke(i)
		self._modified = True

	def append_stroke (self, stroke):
		"""append_stroke (self, stroke)
		append a stroke
		"""
		self._strokes.append (stroke)
		self.queue_draw ()
		self._modified = True
		self.emit ("strokes-changed")
	
	def swap_strokes (self, i):
		"""swap_strokes (self, i)
		swap stroke[i] & stroke[i+1]
		"""
		s = self._strokes[i]
		self._strokes[i] = self._strokes[i+1]
		self._strokes[i+1] = s
		self._modified = True
		self.queue_draw ()
		self.emit ("strokes-changed")
	def set_selection (self, selection):
		self._selection = selection
		self.queue_draw()

	def __draw (self, cr, pg):
		# Draw background
		cr.save ()
		cr.set_source_rgb (1, 1, 1)
		cr.paint ()
		cr.restore ()

		# Draw cross
		cr.save ()
		cr.scale (self._width / SCALE, self._height / SCALE)
		cr.set_source_rgba (*COLOR_CROSS)
		cr.set_line_width (4)
		cr.set_dash ([8, 8], 2)
		cr.move_to (500, 0)
		cr.line_to (500, 1000)
		cr.move_to (0, 500)
		cr.line_to (1000, 500)
		cr.stroke ()
		cr.restore ()

		# Draw charater as background
		cr.save ()
		layout = pango.Layout (pg)
		layout.set_text (self._chars)
		layout.set_font_description (self._font_desc)
		w, h = layout.get_pixel_size ()
		cr.scale (float (self._width) / w, float (self._height) / h)
		cr.move_to (0, 0)
		cr.set_source_rgba (*COLOR_CHAR)
		cr.show_layout (layout)
		cr.restore ()

		#Draw strokes
		cr.save ()
		cr.scale (float (self._width) / SCALE, float (self._height) / SCALE)
		cr.set_line_width (10)
		# cr.set_line_cap (cairo.LINE_CAP_ROUND)
		# cr.set_line_join (cairo.LINE_JOIN_MITER)
		layout.set_font_description (self._font_desc_number)
		i = 0
		for stroke in self._strokes:
			# Skip highlight
			if i not in self._selection:
				# Draw stroke
				cr.move_to (*stroke[0])
				for x, y in stroke[1:]:
					cr.line_to (x, y)
			# Draw stroke number
			i = i + 1
			x1, y1= stroke[0]
			x2, y2= stroke[1]
			cr.save ()
			cr.set_source_rgba (*COLOR_NUMBER)
			layout.set_text (str (i))
			cr.move_to ((x1 * 2 + x2) / 3 + 6, (y1  * 2 + y2) / 3 + 6)
			cr.show_layout (layout)
			cr.restore ()

		cr.stroke ()
		cr.restore ()
		
		# Draw highlight
		for i in self._selection:
			cr.save ()
			cr.scale (float(self._width) / SCALE, float(self._height) / SCALE)
			cr.set_line_width (10)
			stroke = self._strokes[i]
			cr.move_to (*stroke[0])
			for x, y in stroke[1:]:
				cr.line_to (x, y)
			cr.set_source_rgba (*COLOR_HIGHLIGHT)
			cr.stroke ()
			cr.restore ()

		# Draw current stroke
		cr.save ()
		cr.scale (float (self._width) / SCALE, float (self._height) / SCALE)
		stroke_num = len (self._curr_stroke)
		if stroke_num == 1:
			cr.arc(self._curr_stroke[0][0], self._curr_stroke[0][1], 3, 0, 2.0 * math.pi)
		if stroke_num >= 2:
			cr.move_to (*self._curr_stroke[0])
			for x, y in self._curr_stroke[1:]:
				cr.line_to (x, y)
		cr.stroke ()
		cr.restore ()
	
	def __size_allocate_cb (self, widget, rect):
		self._width = rect.width
		self._height = rect.height
	
	def __expose_cb (self, widget, event):
		cr = widget.window.cairo_create ()
		pg = widget.create_pango_context ()
		self.__draw (cr, pg)
		return True

	def __key_press_cb (self, widget, event):
		if event.keyval == keysyms.Escape:
			# Discard current stroke
			self.delete_stroke ()
		elif event.keyval in (keysyms.KP_Enter, keysyms.Return) :
			# Finish current stroke
			if len (self._curr_stroke) >= 2:
				self.append_stroke (self._curr_stroke)
			self._curr_stroke = []
			self.queue_draw ()
				

	def __button_press_cb (self, widget, event):
		self.grab_focus ()
		if event.button == 1:
			if event.type == gdk.BUTTON_PRESS:
				self._curr_stroke.append ((round (event.x * SCALE / self._width), 
					round (event.y * SCALE / self._height)))
				self.queue_draw ()
			elif event.type == gdk._2BUTTON_PRESS:
				# Finish current stroke
				if len (self._curr_stroke) >= 3:
					self._curr_stroke.pop ()
					self.append_stroke (self._curr_stroke)
				self._curr_stroke = []				
		elif event.button == 3:
			# Finish current stroke
			if len (self._curr_stroke) >= 2:
				self.append_stroke (self._curr_stroke)
			self._curr_stroke = []

gobject.type_register (Canvas)
